<?php
require_once 'config/database.php';
require_once 'includes/header.php';

// --- Dashboard Data Queries ---
try {
    // 1. Stat Card Data (Existing Queries)
    $total_students = $pdo->query("SELECT COUNT(student_id) FROM students WHERE is_active = 1")->fetchColumn();
    $today_collection = $pdo->query("SELECT SUM(amount) FROM payments WHERE payment_date = CURDATE()")->fetchColumn() ?? 0;
    $total_dues = $pdo->query("SELECT SUM(amount_due - amount_paid) FROM fee_assignments WHERE status IN ('Pending', 'Partially Paid')")->fetchColumn() ?? 0;
    $monthly_collection = $pdo->query("SELECT SUM(amount) FROM payments WHERE MONTH(payment_date) = MONTH(CURDATE()) AND YEAR(payment_date) = YEAR(CURDATE())")->fetchColumn() ?? 0;

    // --- 2. Data for Monthly Collection Chart (Last 6 Months) ---
    $monthly_chart_data = [];
    $stmt_monthly = $pdo->query("
        SELECT DATE_FORMAT(payment_date, '%b %Y') as month, SUM(amount) as total
        FROM payments
        WHERE payment_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
        GROUP BY YEAR(payment_date), MONTH(payment_date)
        ORDER BY YEAR(payment_date), MONTH(payment_date)
    ");
    while ($row = $stmt_monthly->fetch()) {
        $monthly_chart_data['labels'][] = $row['month'];
        $monthly_chart_data['data'][] = $row['total'];
    }

    // --- 3. Data for Fee Status Pie Chart ---
    $fee_status_data = $pdo->query("
        SELECT status, COUNT(*) as count
        FROM fee_assignments
        GROUP BY status
    ")->fetchAll(PDO::FETCH_KEY_PAIR);

    // --- 4. Data for Recent Payments Log ---
    $recent_payments = $pdo->query("
        SELECT p.amount, p.payment_date, s.full_name
        FROM payments p
        JOIN students s ON p.student_id = s.student_id
        ORDER BY p.created_at DESC
        LIMIT 5
    ")->fetchAll();

} catch(PDOException $e) {
    echo "<div class='alert alert-danger'>Could not fetch all dashboard data. Error: " . $e->getMessage() . "</div>";
}
?>

<style>
/* Grid layout for dashboard widgets */
.dashboard-grid {
    display: grid;
    grid-template-columns: repeat(12, 1fr);
    gap: 20px;
}
.stats-container {
    grid-column: span 12;
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
}
.chart-container {
    grid-column: span 12;
}
.recent-activity {
    grid-column: span 12;
}
@media (min-width: 992px) {
    .chart-container.monthly-chart { grid-column: span 8; }
    .chart-container.status-chart { grid-column: span 4; }
    .recent-activity { grid-column: span 12; }
}

/* Recent payments list styling */
.recent-payments-list { list-style: none; padding: 0; margin: 0; }
.recent-payments-list li { display: flex; justify-content: space-between; align-items: center; padding: 12px 0; border-bottom: 1px solid var(--border-color); }
.recent-payments-list li:last-child { border-bottom: none; }
.recent-payments-list .student-name { font-weight: 600; }
.recent-payments-list .payment-amount { font-weight: bold; color: #28a745; }
.recent-payments-list .payment-date { font-size: 0.9em; color: #6c757d; }

/* === CORRECTED: Chart Container Height Fix === */
.chart-wrapper {
    position: relative;
    height: 350px; /* Define a height for the chart's canvas area */
    margin-top: 15px;
}
</style>

<h1>Dashboard Overview</h1>

<div class="dashboard-grid">
    <!-- Stat Cards Section -->
    <div class="stats-container">
        <div class="stat-card students"><div class="stat-card-icon">👨‍🎓</div><div class="stat-card-info"><h4>Total Students</h4><p><?php echo $total_students; ?></p></div></div>
        <div class="stat-card today-collection"><div class="stat-card-icon">💵</div><div class="stat-card-info"><h4>Today's Collection</h4><p>₹ <?php echo number_format($today_collection, 2); ?></p></div></div>
        <div class="stat-card total-dues"><div class="stat-card-icon">❗</div><div class="stat-card-info"><h4>Total Dues</h4><p>₹ <?php echo number_format($total_dues, 2); ?></p></div></div>
        <div class="stat-card monthly-collection"><div class="stat-card-icon">📅</div><div class="stat-card-info"><h4>This Month</h4><p>₹ <?php echo number_format($monthly_collection, 2); ?></p></div></div>
    </div>

    <!-- Monthly Collection Chart -->
    <div class="chart-container monthly-chart">
        <div class="card">
            <div class="card-body">
                <h3>Monthly Collection Trend (Last 6 Months)</h3>
                <div class="chart-wrapper">
                    <canvas id="monthlyCollectionChart"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- Fee Status Chart -->
    <div class="chart-container status-chart">
        <div class="card">
            <div class="card-body">
                <h3>Fee Status Overview</h3>
                <div class="chart-wrapper">
                    <canvas id="feeStatusChart"></canvas>
                </div>
            </div>
        </div>
    </div>

    <!-- Recent Activity Section -->
    <div class="recent-activity">
        <div class="card">
            <div class="card-body">
                <h3>Recent Payments</h3>
                <ul class="recent-payments-list">
                    <?php if (!empty($recent_payments)): ?>
                        <?php foreach ($recent_payments as $payment): ?>
                            <li>
                                <div>
                                    <div class="student-name"><?php echo htmlspecialchars($payment['full_name']); ?></div>
                                    <div class="payment-date"><?php echo date("d M, Y", strtotime($payment['payment_date'])); ?></div>
                                </div>
                                <div class="payment-amount">₹ <?php echo number_format($payment['amount'], 2); ?></div>
                            </li>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <li>No recent payments found.</li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- Chart.js Library -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', () => {
    // --- Monthly Collection Bar Chart ---
    const monthlyCtx = document.getElementById('monthlyCollectionChart').getContext('2d');
    new Chart(monthlyCtx, {
        type: 'bar',
        data: {
            labels: <?php echo json_encode($monthly_chart_data['labels'] ?? []); ?>,
            datasets: [{
                label: 'Collection Amount',
                data: <?php echo json_encode($monthly_chart_data['data'] ?? []); ?>,
                backgroundColor: 'rgba(52, 152, 219, 0.7)',
                borderColor: 'rgba(52, 152, 219, 1)',
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: { beginAtZero: true, ticks: { callback: value => '₹ ' + value } }
            }
        }
    });

    // --- Fee Status Doughnut Chart ---
    const statusCtx = document.getElementById('feeStatusChart').getContext('2d');
    new Chart(statusCtx, {
        type: 'doughnut',
        data: {
            labels: ['Pending', 'Paid', 'Partially Paid'],
            datasets: [{
                data: [
                    <?php echo $fee_status_data['Pending'] ?? 0; ?>,
                    <?php echo $fee_status_data['Paid'] ?? 0; ?>,
                    <?php echo $fee_status_data['Partially Paid'] ?? 0; ?>
                ],
                backgroundColor: ['#e74c3c', '#2ecc71', '#f1c40f'],
            }]
        },
        options: { responsive: true, maintainAspectRatio: false }
    });
});
</script>

<?php require_once 'includes/footer.php'; ?>

