<?php
require_once 'config/database.php';
require_once 'includes/header.php';

// Fetch all classes for the filter dropdown
try {
    $classes = $pdo->query("SELECT class_id, class_name, section FROM classes ORDER BY class_name, section")->fetchAll();
} catch (PDOException $e) {
    die("Could not fetch classes.");
}

// Check if a class is selected for filtering
$selected_class_id = $_GET['class_id'] ?? 'all';
$dues = [];
$total_balance = 0;

try {
    // This is the core query for the report
    $sql = "
        SELECT 
            s.student_id,
            s.student_uid,
            s.full_name,
            s.guardian_contact,
            SUM(fa.amount_due) AS total_due,
            SUM(fa.amount_paid) AS total_paid,
            (SUM(fa.amount_due) - SUM(fa.amount_paid)) AS balance
        FROM students s
        JOIN fee_assignments fa ON s.student_id = fa.student_id
    ";

    // Add a WHERE clause if a specific class is selected
    if ($selected_class_id != 'all' && is_numeric($selected_class_id)) {
        $sql .= " WHERE s.class_id = :class_id";
    }

    $sql .= "
        GROUP BY s.student_id
        HAVING balance > 0
        ORDER BY s.full_name ASC
    ";

    $stmt = $pdo->prepare($sql);

    // Bind the class_id parameter if it exists
    if ($selected_class_id != 'all' && is_numeric($selected_class_id)) {
        $stmt->bindParam(':class_id', $selected_class_id, PDO::PARAM_INT);
    }
    
    $stmt->execute();
    $dues = $stmt->fetchAll();

    // Calculate the grand total balance
    foreach($dues as $due) {
        $total_balance += $due['balance'];
    }

} catch(PDOException $e) {
    echo "<div class='alert alert-danger'>Database Error: " . $e->getMessage() . "</div>";
}
?>

<div class="content-header">
    <h1>Student Dues Report</h1>
    <a href="reports.php" class="btn">⬅️ Back to Reports Menu</a>
</div>

<div class="card">
    <div class="card-body">
        <form method="GET" action="dues_report.php" class="filter-form">
            <div class="form-group">
                <label for="class_id">Filter by Class:</label>
                <select name="class_id" id="class_id">
                    <option value="all">All Classes</option>
                    <?php foreach ($classes as $class): ?>
                        <option value="<?php echo $class['class_id']; ?>" <?php echo ($selected_class_id == $class['class_id']) ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($class['class_name'] . ' - ' . $class['section']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="btn btn-primary">View Report</button>
        </form>
    </div>
</div>

<div class="card" style="margin-top:20px;">
    <div class="card-body">
        <div class="report-header">
            <h3>Outstanding Balances</h3>
            <div class="total-display">
                Total Outstanding: <span>₹ <?php echo number_format($total_balance, 2); ?></span>
            </div>
        </div>

        <table class="data-table">
            <thead>
                <tr>
                    <th>Student UID</th>
                    <th>Student Name</th>
                    <th>Guardian Contact</th>
                    <th>Total Due (₹)</th>
                    <th>Total Paid (₹)</th>
                    <th>Balance (₹)</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($dues) > 0): ?>
                    <?php foreach ($dues as $row): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($row['student_uid']); ?></td>
                            <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                            <td><?php echo htmlspecialchars($row['guardian_contact']); ?></td>
                            <td style="text-align:right;"><?php echo number_format($row['total_due'], 2); ?></td>
                            <td style="text-align:right;"><?php echo number_format($row['total_paid'], 2); ?></td>
                            <td style="text-align:right; font-weight:bold; color:#dc3545;"><?php echo number_format($row['balance'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="6" style="text-align:center;">No outstanding dues found for the selected criteria.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<style>
/* Using existing styles from daily_collection_report.php */
.total-display span { color: #dc3545; } /* Red for dues */
</style>

<?php
require_once 'includes/footer.php';
?>