<?php
require_once 'config/database.php';
require_once 'includes/header.php';

// 1. Validate the Receipt ID from the URL
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    echo "<div class='alert alert-danger'>Invalid receipt ID.</div>";
    require_once 'includes/footer.php';
    exit();
}
$receipt_id = $_GET['id'];

try {
    // 2. Fetch the receipt data from the database
    $stmt = $pdo->prepare("SELECT receipt_data FROM receipts WHERE receipt_id = ?");
    $stmt->execute([$receipt_id]);
    $receipt_record = $stmt->fetch();

    if (!$receipt_record) {
        throw new Exception("Receipt not found.");
    }
    
    // 3. Decode the JSON data into a PHP object
    $data = json_decode($receipt_record['receipt_data']);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Failed to parse receipt data.");
    }
    
    // 4. Fetch dynamic school settings (CORRECTED QUERY)
    $settings = $pdo->query("SELECT setting_key, setting_value FROM settings")->fetchAll(PDO::FETCH_KEY_PAIR);

} catch (Exception $e) {
    echo "<div class='alert alert-danger'>Error: " . $e->getMessage() . "</div>";
    require_once 'includes/footer.php';
    exit();
}

// Helper function to convert number to words
function numberToWords($number) {
    // A simple implementation for demonstration.
    // For a production app, a more robust library is recommended.
    if (!class_exists('NumberFormatter')) {
        return "NumberFormatter class not found. Please enable the intl extension in your php.ini.";
    }
    $formatter = new NumberFormatter("en", NumberFormatter::SPELLOUT);
    return ucwords($formatter->format($number));
}
?>

<style>
    .receipt-container { 
        max-width: 800px; 
        margin: 20px auto; 
        background: #fff; 
        border: 1px solid #ccc; 
        padding: 30px; 
        font-family: 'Arial', sans-serif; 
        color: #333;
    }
    .receipt-header { 
        text-align: center; 
        margin-bottom: 20px; 
        border-bottom: 2px solid #333; 
        padding-bottom: 10px; 
    }
    .receipt-header h2 { margin: 0; font-size: 2rem; }
    .receipt-header p { margin: 5px 0 0; font-size: 0.9rem; }
    .receipt-title { 
        text-align: center; 
        margin-bottom: 25px; 
        font-weight: bold; 
        font-size: 1.2rem; 
        text-decoration: underline; 
        text-transform: uppercase;
    }
    .details-grid { 
        display: grid; 
        grid-template-columns: 1fr 1fr; 
        gap: 20px; 
        margin-bottom: 25px; 
        border-bottom: 1px dashed #ccc;
        padding-bottom: 15px;
    }
    .details-grid div { line-height: 1.6; font-size: 0.9rem; }
    .details-grid span { font-weight: bold; color: #555; }
    .line-items table { width: 100%; border-collapse: collapse; }
    .line-items th, .line-items td { border: 1px solid #ccc; padding: 10px; text-align: left; }
    .line-items th { background-color: #f2f2f2; }
    .total-section { text-align: right; margin-top: 20px; font-size: 1.1rem; }
    .amount-in-words { margin-top: 10px; font-style: italic; font-size: 0.9rem; }
    .receipt-footer { 
        margin-top: 40px; 
        padding-top: 20px;
        border-top: 2px solid #333;
        display: flex; 
        justify-content: space-between; 
        align-items: flex-end; 
    }
    .print-button-container { text-align: center; margin: 20px; }
    
    @media print {
        body * { visibility: hidden; }
        .sidebar, .header, .print-button-container, .content-header, .footer { display: none !important; }
        .receipt-container, .receipt-container * { visibility: visible; }
        .receipt-container { 
            position: absolute; 
            left: 0; 
            top: 0; 
            width: 100%; 
            margin: 0; 
            padding: 5px; 
            border: none; 
            box-shadow: none;
        }
    }
</style>

<div class="print-button-container">
    <a href="receipts.php" class="btn">Back to Receipt Search</a>
    <button onclick="window.print()" class="btn btn-primary">🖨️ Print Receipt</button>
</div>

<div class="receipt-container">
    <div class="receipt-header">
        <h2><?php echo htmlspecialchars($settings['school_name'] ?? 'School Name'); ?></h2>
        <p><?php echo htmlspecialchars($settings['school_address'] ?? 'School Address'); ?></p>
        <p><?php echo htmlspecialchars($settings['school_contact'] ?? 'Contact Info'); ?></p>
    </div>

    <div class="receipt-title">Fee Receipt</div>

    <div class="details-grid">
        <div>
            <span>Receipt No:</span> <?php echo htmlspecialchars($data->receipt_no); ?><br>
            <span>Payment ID:</span> <?php echo htmlspecialchars($data->payment_id); ?><br>
            <span>Date:</span> <?php echo date("d F, Y", strtotime($data->date)); ?>
        </div>
        <div>
            <span>Student Name:</span> <?php echo htmlspecialchars($data->student_name); ?><br>
            <span>Student UID:</span> <?php echo htmlspecialchars($data->student_uid); ?>
        </div>
    </div>

    <div class="line-items">
        <table>
            <thead>
                <tr>
                    <th>Sr. No.</th>
                    <th>Particulars</th>
                    <th style="text-align: right;">Amount Paid (₹)</th>
                </tr>
            </thead>
            <tbody>
                <?php $i = 1; foreach ($data->details as $item): ?>
                <tr>
                    <td><?php echo $i++; ?></td>
                    <td><?php echo htmlspecialchars($item->title); ?></td>
                    <td style="text-align: right;"><?php echo number_format($item->amount_paid, 2); ?></td>
                </tr>
                <?php endforeach; ?>
                 <tr>
                    <td colspan="2" style="text-align: right; font-weight: bold;">Total Paid</td>
                    <td style="text-align: right; font-weight: bold;"><?php echo number_format($data->total_paid, 2); ?></td>
                </tr>
            </tbody>
        </table>
    </div>

    <div class="total-section">
        <div class="amount-in-words">
            <strong>In Words:</strong> <?php echo htmlspecialchars(numberToWords($data->total_paid)); ?> Rupees Only
        </div>
        
        <?php if (isset($data->remaining_balance) && $data->remaining_balance > 0): ?>
        <div style="margin-top: 15px; color: #dc3545; font-weight: bold;">
            Total Outstanding Balance after this payment: ₹ <?php echo number_format($data->remaining_balance, 2); ?>
        </div>
        <?php endif; ?>
    </div>
    
    <div class="receipt-footer">
        <div>
            <strong>Mode:</strong> <?php echo htmlspecialchars($data->payment_mode); ?>
            <?php if (!empty($data->ref_no)): ?>
                <br><span>Ref No:</span> <?php echo htmlspecialchars($data->ref_no); ?>
            <?php endif; ?>
        </div>
        <div style="text-align: center;">
            <br><br>
            <strong style="display: block; border-top: 1px dotted #555; padding-top: 5px;"><?php echo htmlspecialchars($data->collected_by_name); ?></strong>
            (Authorized Signatory)
        </div>
    </div>
</div>

<?php
require_once 'includes/footer.php';
?>
